
/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright 2009--2026 by Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


#pragma once


/////////////////////// Stdlib includes
#include <limits>
#include <vector>
#include <memory>


/////////////////////// Qt includes
#include <QDomElement>


/////////////////////// pappsomspp includes


/////////////////////// Local includes

// For MSXPS_REGISTER_JS_CLASS(NS_IDENT, CLASS_NAME)
// At the end of the file.
#include "MsXpS/libXpertMassCore/jsclassregistrar.h"

#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/Utils.hpp"
#include "Isotope.hpp"
#include "IsotopicData.hpp"

namespace MsXpS
{
namespace libXpertMassCore
{
/*  BEGIN CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: Formula
 */
class DECLSPEC Formula: public QObject
{
  Q_OBJECT
  Q_PROPERTY(QString title READ getTitle WRITE setTitle NOTIFY titleChanged)
  Q_PROPERTY(QString actionFormula READ getActionFormula WRITE setActionFormula
               NOTIFY actionFormulaChanged)
  Q_PROPERTY(bool valid READ isValid NOTIFY validChanged)

  public:
  enum class SplitResult : uint8_t
  {
    NOT_SET             = 0x0000,
    FAILURE             = 1 << 0, //!< The splitting work failed.
    HAS_PLUS_COMPONENT  = 1 << 1, //!< The action formula a plus component
    HAS_MINUS_COMPONENT = 1 << 2, //!< The action formula a minus component
    HAS_BOTH_COMPONENTS =
      (HAS_PLUS_COMPONENT |
       HAS_MINUS_COMPONENT) //!< The action formula had both component types
  };
  Q_ENUM(SplitResult)


  Q_INVOKABLE explicit Formula(QObject *parent = nullptr);
  explicit Formula(const QDomElement &element,
                   int version     = 1,
                   QObject *parent = nullptr);
  Q_INVOKABLE explicit Formula(const QString &formula_string,
                               QObject *parent = nullptr);

  // Pseudo copy constructor
  explicit Formula(const Formula &other, QObject *parent = nullptr);

  virtual ~Formula();

  Q_INVOKABLE Formula *clone(const Formula &other, QObject *parent = nullptr);

  Q_INVOKABLE Formula &initialize(const Formula &other);

  //////////////// THE ACTIONFORMULA /////////////////////
  Q_INVOKABLE void setActionFormula(const QString &formula);
  Q_INVOKABLE void setActionFormula(const Formula &formula);
  Q_INVOKABLE bool appendActionFormula(const QString &formula);
  Q_INVOKABLE QString getActionFormula(bool withTitle = false) const;

  //////////////// THE TITLE /////////////////////
  Q_INVOKABLE void setTitle(const QString &title);
  Q_INVOKABLE QString getTitle() const;
  Q_INVOKABLE QString extractTitle() const;
  Q_INVOKABLE QString removeTitle();

  //////////////// THE SYNTAX CHECKING LOGIC /////////////////////
  Q_INVOKABLE bool checkSyntax() const;
  Q_INVOKABLE static bool checkSyntax(const QString &formula_string);

  //////////////// OPERATORS /////////////////////
  virtual Formula &operator=(const Formula &other);
  virtual bool operator==(const Formula &other) const;
  virtual bool operator!=(const Formula &other) const;

  //////////////// THE SUB-FORMULAS OPERATIONS /////////////////////
  SplitResult splitActionParts(IsotopicDataCstSPtr isotopic_data_csp,
                               QString &plus_formula,
                               QString &minus_formula,
                               std::map<QString, double> &symbol_count_map,
                               double times = 1,
                               bool store   = false,
                               bool reset   = false) const;
  SplitResult splitActionParts(IsotopicDataCstSPtr isotopic_data_csp,
                               double times = 1,
                               bool store   = false,
                               bool reset   = false);
  Q_INVOKABLE static QChar actions(const QString &formula);
  Q_INVOKABLE QChar actions() const;
  Q_INVOKABLE bool hasNetMinusPart();
  Q_INVOKABLE QString getPlusFormula() const;
  Q_INVOKABLE QString getMinusFormula() const;

  //////////////// VALIDATIONS /////////////////////
  virtual bool validate(IsotopicDataCstSPtr isotopic_data_csp,
                        ErrorList *error_list_p) const;
  virtual bool validate(IsotopicDataCstSPtr isotopic_data_csp,
                        bool store,
                        bool reset,
                        ErrorList *error_list_p);
  virtual bool isValid() const;

  //////////////// THE SYMBOLS-COUNT OPERATIONS /////////////////////
  const std::map<QString, double> &getSymbolCountMapCstRef() const;
  double symbolCount(const QString &symbol) const;
  bool accountSymbolCounts(IsotopicDataCstSPtr isotopic_data_csp, int times);
  double accountSymbolCountPair(std::map<QString, double> &symbol_count_map,
                                const QString &symbol,
                                double count = 1) const;
  double accountSymbolCountPair(const QString &symbol, double count = 1);
  std::size_t accountFormula(const QString &formula_string,
                             IsotopicDataCstSPtr isotopic_data_csp,
                             double times,
                             bool &ok);

  //////////////// ELEMENTAL COMPOSITION /////////////////////
  Q_INVOKABLE QString elementalComposition(
    std::vector<std::pair<QString, double>> *symbol_count_pairs_p =
      nullptr) const;

  //////////////// MASS OPERATIONS /////////////////////
  Formula &accountMasses(bool &ok,
                         IsotopicDataCstSPtr isotopic_data_csp,
                         double &mono,
                         double &avg,
                         double times = 1);
  static Formula &accountMasses(Formula &formula,
                                bool &ok,
                                IsotopicDataCstSPtr isotopic_data_csp,
                                double &mono,
                                double &avg,
                                double times = 1);

  //////////////// XML DATA LOADING WRITING /////////////////////
  bool renderXmlFormulaElement(const QDomElement &element, int version = 1);
  Q_INVOKABLE QString formatXmlFormulaElement(
    int offset, const QString &indent = Utils::xmlIndentationToken);

  //////////////// UTILS /////////////////////
  int removeSpaces();
  double totalAtoms() const;
  double totalIsotopes(IsotopicDataCstSPtr isotopic_data_csp) const;
  void clear();

  static void registerJsConstructor(QJSEngine *engine);

  signals:
  void titleChanged();
  void actionFormulaChanged();
  void validChanged();

  protected:
  QString m_title;
  QString m_actionFormula;
  QString m_plusFormula;
  QString m_minusFormula;

  // Map that relates the chemical element symbols encounted in formula(e) to
  // their count in the formula(e). The count can be a double or an int.
  std::map<QString, double> m_symbolCountMap;

  mutable bool m_isValid = false;

  //////////////// PRIVATE FUNCTIONS /////////////////////
  //////////////// PRIVATE FUNCTIONS /////////////////////
  //////////////// PRIVATE FUNCTIONS /////////////////////
  private:
  void setPlusFormula(const QString &formula);
  void setMinusFormula(const QString &formula);
};

// Overload bitwise OR operator
inline Formula::SplitResult
operator|(Formula::SplitResult lhs, Formula::SplitResult rhs)
{
  return static_cast<Formula::SplitResult>(
    static_cast<std::underlying_type_t<Formula::SplitResult>>(lhs) |
    static_cast<std::underlying_type_t<Formula::SplitResult>>(rhs));
}

// Overload bitwise AND operator
inline Formula::SplitResult
operator&(Formula::SplitResult lhs, Formula::SplitResult rhs)
{
  return static_cast<Formula::SplitResult>(
    static_cast<std::underlying_type_t<Formula::SplitResult>>(lhs) &
    static_cast<std::underlying_type_t<Formula::SplitResult>>(rhs));
}

// Overload bitwise XOR operator
inline Formula::SplitResult
operator^(Formula::SplitResult lhs, Formula::SplitResult rhs)
{
  return static_cast<Formula::SplitResult>(
    static_cast<std::underlying_type_t<Formula::SplitResult>>(lhs) ^
    static_cast<std::underlying_type_t<Formula::SplitResult>>(rhs));
}

// Overload bitwise NOT operator
inline Formula::SplitResult
operator~(Formula::SplitResult opt)
{
  return static_cast<Formula::SplitResult>(
    ~static_cast<std::underlying_type_t<Formula::SplitResult>>(opt));
}

// Overload bitwise OR assignment operator
inline Formula::SplitResult &
operator|=(Formula::SplitResult &lhs, Formula::SplitResult rhs)
{
  lhs = lhs | rhs;
  return lhs;
}

// Overload bitwise AND assignment operator
inline Formula::SplitResult &
operator&=(Formula::SplitResult &lhs, Formula::SplitResult rhs)
{
  lhs = lhs & rhs;
  return lhs;
}

/*  END CLASS JS REFERENCE
 *  namespace: MsXpS::libXpertMassCore
 *  class name: Formula
 */

} // namespace libXpertMassCore

MSXPS_REGISTER_JS_CLASS(MsXpS::libXpertMassCore, Formula)

} // namespace MsXpS
