--
-- This file is part of TALER
-- Copyright (C) 2026 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--


DROP FUNCTION IF EXISTS merchant_do_account_kyc_get_status;
CREATE FUNCTION merchant_do_account_kyc_get_status (
  IN in_merchant_id TEXT,
  IN in_now INT8,
  IN in_exchange_url TEXT,  -- can be NULL
  IN in_h_wire BYTEA  -- can be NULL
) RETURNS TABLE (
  out_h_wire BYTEA, -- never NULL
  out_payto_uri TEXT, -- never NULL
  out_exchange_url TEXT,
  out_kyc_timestamp INT8,
  out_kyc_ok BOOLEAN,
  out_access_token BYTEA,
  out_exchange_http_status INT4,
  out_exchange_ec_code INT4,
  out_aml_review BOOLEAN,
  out_jaccount_limits TEXT
)
LANGUAGE plpgsql
AS $$
DECLARE
  my_merchant_serial INT8;
  my_account_serial INT8;
  my_h_wire BYTEA;
  my_payto_uri TEXT;
  my_kyc_record RECORD;

BEGIN
  -- Get the merchant_serial from merchant_instances
  SELECT merchant_serial
    INTO my_merchant_serial
    FROM merchant_instances
   WHERE merchant_id = in_merchant_id;
  IF NOT FOUND
  THEN
    RETURN;
  END IF;

  -- Iterate over merchant_accounts for this merchant
  FOR my_account_serial, my_h_wire, my_payto_uri
  IN SELECT account_serial, h_wire, payto_uri
    FROM merchant_accounts
    WHERE merchant_serial = my_merchant_serial
      AND active
      AND (in_h_wire IS NULL OR h_wire = in_h_wire)
  LOOP

    -- Fetch KYC info for this account (can have multiple results)
    FOR my_kyc_record IN
      SELECT
        mk.kyc_serial_id
       ,mk.exchange_url
       ,mk.kyc_timestamp
       ,mk.kyc_ok
       ,mk.access_token
       ,mk.exchange_http_status
       ,mk.exchange_ec_code
       ,mk.aml_review
       ,mk.jaccount_limits::TEXT
      FROM merchant_kyc mk
      WHERE mk.account_serial = my_account_serial
        AND (in_exchange_url IS NULL OR mk.exchange_url = in_exchange_url)
    LOOP
      -- Ask taler-merchant-kyccheck to get us an update on the status ASAP
      UPDATE merchant_kyc
         SET next_kyc_poll=in_now
       WHERE kyc_serial_id = my_kyc_record.kyc_serial_id;
      NOTIFY XDQM4Z4N0D3GX0H9JEXH70EBC2T3KY7HC0TJB0Z60D2H781RXR6AG; -- MERCHANT_EXCHANGE_KYC_UPDATE_FORCED
      RETURN QUERY
      SELECT
        my_h_wire,
        my_payto_uri,
        my_kyc_record.exchange_url,
        my_kyc_record.kyc_timestamp,
        my_kyc_record.kyc_ok,
        my_kyc_record.access_token,
        my_kyc_record.exchange_http_status,
        my_kyc_record.exchange_ec_code,
        my_kyc_record.aml_review,
        my_kyc_record.jaccount_limits::TEXT;
    END LOOP; -- loop over exchanges with KYC status for the given account

    IF NOT FOUND
    THEN
      -- Still return to server that we do NOT know anything
      -- for the given exchange yet (but that the bank account exists)
      RETURN QUERY
      SELECT
        my_h_wire,
        my_payto_uri,
        NULL::TEXT,
        NULL::INT8,
        NULL::BOOLEAN,
        NULL::BYTEA,
        NULL::INT4,
        NULL::INT4,
        NULL::BOOLEAN,
        NULL::TEXT;
    END IF;

  END LOOP; -- loop over merchant_accounts

END $$;
COMMENT ON FUNCTION merchant_do_account_kyc_get_status
  IS 'Returns the KYC status of selected exchanges and accounts, but ALSO resets the next_kyc_check time for all returned data points to the current time (in_now argument)';
