/*
  This file is part of TALER
  Copyright (C) 2024, 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-exchange-httpd_aml-attributes-get.c
 * @brief Return summary information about KYC attributes
 * @author Christian Grothoff
 */
#include "taler/platform.h"
#include <gnunet/gnunet_util_lib.h>
#include <jansson.h>
#include <microhttpd.h>
#include <pthread.h>
#include "taler/taler_json_lib.h"
#include "taler/taler_mhd_lib.h"
#include "taler/taler_kyclogic_lib.h"
#include "taler/taler_signatures.h"
#include "taler-exchange-httpd.h"
#include "taler/taler_exchangedb_plugin.h"
#include "taler-exchange-httpd_aml-attributes-get.h"
#include "taler-exchange-httpd_metrics.h"

/**
 * Maximum number of records we return in one request.
 */
#define MAX_RECORDS 1024


/**
 * Closure for the detail_cb().
 */
struct ResponseContext
{
  /**
   * Format of the response we are to generate.
   */
  enum
  {
    RCF_JSON,
    RCF_PDF
  } format;

  /**
   * Stored in a DLL while suspended.
   */
  struct ResponseContext *next;

  /**
   * Stored in a DLL while suspended.
   */
  struct ResponseContext *prev;

  /**
   * Context for this request.
   */
  struct TEH_RequestContext *rc;

  /**
   * Async context used to run Typst.
   */
  struct TALER_MHD_TypstContext *tc;

  /**
   * Response to return.
   */
  struct MHD_Response *response;

  /**
   * HTTP status to use with @e response.
   */
  unsigned int http_status;

  /**
   * True if this is for a wallet.
   */
  bool is_wallet;

  /**
   * Where we store the response data.
   */
  union
  {
    /**
     * If @e format is #RCF_JSON.
     */
    json_t *json;

    /**
     * If @e format is #RCF_PDF.
     */
    struct
    {

      /**
       * Typst forms to compile.
       */
      struct TALER_MHD_TypstDocument docs[MAX_RECORDS];

      /**
       * JSON data for each Typst form.
       */
      json_t *jdata[MAX_RECORDS];

      /**
       * Next write offset into @e docs and @e jdata.
       */
      size_t off;

      /**
       * Global attributes we need to inject in to each @e jdata.
       */
      json_t *global_attrs;

    } pdf;

  } details;
};


/**
 * DLL of requests awaiting Typst.
 */
static struct ResponseContext *rctx_head;

/**
 * DLL of requests awaiting Typst.
 */
static struct ResponseContext *rctx_tail;


void
TEH_handler_aml_attributes_get_cleanup ()
{
  struct ResponseContext *rctx;

  while (NULL != (rctx = rctx_head))
  {
    GNUNET_CONTAINER_DLL_remove (rctx_head,
                                 rctx_tail,
                                 rctx);
    MHD_resume_connection (rctx->rc->connection);
  }
}


/**
 * Free resources from @a rc
 *
 * @param[in] rc context to clean up
 */
static void
free_rc (struct TEH_RequestContext *rc)
{
  struct ResponseContext *rctx = rc->rh_ctx;

  if (NULL != rctx->tc)
  {
    TALER_MHD_typst_cancel (rctx->tc);
    rctx->tc = NULL;
  }
  if (NULL != rctx->response)
  {
    MHD_destroy_response (rctx->response);
    rctx->response = NULL;
  }
  switch (rctx->format)
  {
  case RCF_JSON:
    json_decref (rctx->details.json);
    break;
  case RCF_PDF:
    for (size_t i = 0; i<rctx->details.pdf.off; i++)
    {
      json_decref (rctx->details.pdf.jdata[i]);
      rctx->details.pdf.jdata[i] = NULL;
    }
    json_decref (rctx->details.pdf.global_attrs);
    rctx->details.pdf.global_attrs = NULL;
    break;
  }
  GNUNET_free (rctx);
}


/**
 * Dump attachment @a attach into @a rc.
 *
 * @param[in,out] rc response context to update
 * @param attach attachment to dump
 * @return true on success
 */
static bool
dump_attachment (struct ResponseContext *rc,
                 const json_t *attach)
{
  if (! json_is_string (attach))
    return false;
  if (rc->details.pdf.off >= MAX_RECORDS)
  {
    GNUNET_break (0);
    return false;
  }
  rc->details.pdf.jdata[rc->details.pdf.off]
    = json_incref ((json_t *) attach);
  rc->details.pdf.docs[rc->details.pdf.off].form_name
    = NULL; /* inline PDF */
  rc->details.pdf.docs[rc->details.pdf.off].data
    = rc->details.pdf.jdata[rc->details.pdf.off];
  rc->details.pdf.off++;
  return true;
}


/**
 * Recursively scan @a attrs for ".FILE" members and dump those
 * attachments into @a rc.
 *
 * @param[in,out] rc response context to update
 * @param attrs attributes to scan recursively
 * @return true on success
 */
static bool
dump_attachments (struct ResponseContext *rc,
                  const json_t *attrs)
{
  bool ret = true;

  if (json_is_array (attrs))
  {
    const json_t *e;
    size_t i;

    json_array_foreach ((json_t *) attrs, i, e)
    if (! dump_attachments (rc,
                            e))
      ret = false;
  }
  if (json_is_object (attrs))
  {
    const json_t *e;
    const char *k;

    json_object_foreach ((json_t *) attrs, k, e)
    {
      if (0 == strcmp (k,
                       "CONTENTS"))
      {
        const char *v;

        /* Make sure this is the supported encoding */
        v = json_string_value (json_object_get (attrs,
                                                "ENCODING"));
        if (NULL == v)
        {
          GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                      "`CONTENTS' attribute found without `ENCODING', skipping dumping attachment\n");
          continue; /* maybe not a file!? */
        }
        if (0 != strcmp (v,
                         "base64"))
        {
          GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                      "Attachment with unsupported encoding `%s' encountered, skipping attachment in PDF generation\n",
                      v);
          continue;
        }
        v = json_string_value (json_object_get (attrs,
                                                "MIME_TYPE"));
        if (NULL == v)
        {
          GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                      "`CONTENTS' attribute found without `MIME_TYPE', skipping dumping attachment\n");
          continue;
        }
        if (0 != strcmp (v,
                         "application/pdf"))
        {
          GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                      "Attachment with unsupported mime type `%s' encountered, skipping attachment in PDF generation\n",
                      v);
          continue;
        }
        /* Filename is optional, only log it */
        v = json_string_value (json_object_get (attrs,
                                                "FILENAME"));
        if (NULL != v)
        {
          GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                      "Dumping attachment `%s'\n",
                      v);
        }
        if (! dump_attachment (rc,
                               e))
          ret = false;
      }
      else if (! dump_attachments (rc,
                                   e))
        ret = false;
    }
  }
  return ret;
}


/**
 * Return AML account attributes.
 *
 * @param cls closure
 * @param row_id current row in kyc_attributes table
 * @param collection_time when were the attributes collected
 * @param by_aml_officer was the attribute set filed by the AML officer
 * @param staff_name name of the officer, NULL if not @a by_aml_officer
 * @param enc_attributes_size length of @a enc_attributes
 * @param enc_attributes the encrypted collected attributes
 */
static void
detail_cb (
  void *cls,
  uint64_t row_id,
  struct GNUNET_TIME_Timestamp collection_time,
  bool by_aml_officer,
  const char *staff_name,
  size_t enc_attributes_size,
  const void *enc_attributes)
{
  static char *datadir = NULL;
  struct ResponseContext *rc = cls;
  json_t *attrs;
  const char *form_id;

  if (NULL == datadir)
  {
    datadir = GNUNET_OS_installation_get_path (
      TALER_EXCHANGE_project_data (),
      GNUNET_OS_IPK_DATADIR);
  }
  attrs = TALER_CRYPTO_kyc_attributes_decrypt (&TEH_attribute_key,
                                               enc_attributes,
                                               enc_attributes_size);
  if (NULL == attrs)
  {
    GNUNET_break (0);
    return;
  }
  form_id = json_string_value (json_object_get (attrs,
                                                "FORM_ID"));
  if (NULL == form_id)
  {
    GNUNET_break (0);
    return;
  }
  switch (rc->format)
  {
  case RCF_JSON:
    GNUNET_assert (
      0 ==
      json_array_append_new (
        rc->details.json,
        GNUNET_JSON_PACK (
          GNUNET_JSON_pack_int64 ("rowid",
                                  row_id),
          GNUNET_JSON_pack_bool ("by_aml_officer",
                                 by_aml_officer),
          GNUNET_JSON_pack_allow_null (
            GNUNET_JSON_pack_object_steal ("attributes",
                                           attrs)),
          GNUNET_JSON_pack_timestamp ("collection_time",
                                      collection_time)
          )));
    break;
  case RCF_PDF:
    if (rc->details.pdf.off >= MAX_RECORDS)
    {
      GNUNET_break (0);
      return;
    }
    GNUNET_assert (0 ==
                   json_object_set_new (attrs,
                                        "DATADIR",
                                        json_string (datadir)));
    GNUNET_assert (0 ==
                   json_object_set_new (attrs,
                                        "BY_AML_OFFICER",
                                        json_boolean (by_aml_officer)));
    GNUNET_assert (0 ==
                   json_object_set_new (attrs,
                                        "FILING_DATE",
                                        json_string (
                                          GNUNET_STRINGS_timestamp_to_string (
                                            collection_time))));
    if (by_aml_officer)
      GNUNET_assert (0 ==
                     json_object_set_new (attrs,
                                          "AML_STAFF_NAME",
                                          json_string (staff_name)));
    {
      char *have;

      GNUNET_asprintf (&have,
                       "HAVE_%s",
                       form_id);
      GNUNET_assert (0 ==
                     json_object_set_new (rc->details.pdf.global_attrs,
                                          have,
                                          json_true ()));
      GNUNET_free (have);
    }
    rc->details.pdf.jdata[rc->details.pdf.off]
      = attrs;
    rc->details.pdf.docs[rc->details.pdf.off].form_name
      = form_id;
    rc->details.pdf.docs[rc->details.pdf.off].data
      = rc->details.pdf.jdata[rc->details.pdf.off];
    rc->details.pdf.off++;
    if (! dump_attachments (rc,
                            attrs))
    {
      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                  "Failed to dump some attachment!\n");
    }
    break;
  }
}


/**
 * Function called with the result of a #TALER_MHD_typst() operation.
 *
 * @param cls closure
 * @param tr result of the operation
 */
static void
pdf_cb (void *cls,
        const struct TALER_MHD_TypstResponse *tr)
{
  struct ResponseContext *rctx = cls;

  rctx->tc = NULL;
  GNUNET_CONTAINER_DLL_remove (rctx_head,
                               rctx_tail,
                               rctx);
  MHD_resume_connection (rctx->rc->connection);
  TALER_MHD_daemon_trigger ();
  if (TALER_EC_NONE != tr->ec)
  {
    rctx->http_status
      = TALER_ErrorCode_get_http_status (tr->ec);
    rctx->response
      = TALER_MHD_make_error (tr->ec,
                              tr->details.hint);
    return;
  }
  rctx->http_status
    = MHD_HTTP_OK;
  rctx->response
    = TALER_MHD_response_from_pdf_file (tr->details.filename);
}


/**
 * Function called with the latest AML decision on an
 * account. Used to build the cover page.
 *
 * @param cls a `struct ResponseContext *`
 * @param outcome_serial_id row ID of the decision
 * @param decision_time when was the decision taken
 * @param justification what was the given justification
 * @param decider_pub which key signed the decision
 * @param jproperties what are the new account properties
 * @param jnew_rules what are the new account rules
 * @param to_investigate should AML staff investigate
 *          after the decision
 * @param is_active is this the active decision
 */
static void
build_cover_page (
  void *cls,
  uint64_t outcome_serial_id,
  struct GNUNET_TIME_Timestamp decision_time,
  const char *justification,
  const struct TALER_AmlOfficerPublicKeyP *decider_pub,
  const json_t *jproperties,
  const json_t *jnew_rules,
  bool to_investigate,
  bool is_active)
{
  struct ResponseContext *rc = cls;
  json_t *cover;
  json_t *defr = NULL;

  if (NULL == jnew_rules)
  {
    defr = TALER_KYCLOGIC_get_default_legi_rules (rc->is_wallet);
    jnew_rules = defr;
  }
  cover = GNUNET_JSON_PACK (
    GNUNET_JSON_pack_allow_null (
      GNUNET_JSON_pack_object_incref ("properties",
                                      (json_t *) jproperties)),
    GNUNET_JSON_pack_allow_null (
      GNUNET_JSON_pack_object_incref ("rules",
                                      (json_t *) jnew_rules)),
    GNUNET_JSON_pack_allow_null (
      GNUNET_JSON_pack_string ("last_justification",
                               justification)),
    GNUNET_JSON_pack_bool ("is_active",
                           is_active),
    GNUNET_JSON_pack_bool ("to_investigate",
                           to_investigate));
  if (NULL != defr)
    json_decref (defr);
  /* first page, so we really cannot have hit the maximum yet */
  GNUNET_assert (rc->details.pdf.off < MAX_RECORDS);
  rc->details.pdf.jdata[rc->details.pdf.off]
    = cover;
  rc->details.pdf.docs[rc->details.pdf.off].form_name
    = "_cover_";
  rc->details.pdf.docs[rc->details.pdf.off].data
    = rc->details.pdf.jdata[rc->details.pdf.off];
  rc->details.pdf.off++;
}


MHD_RESULT
TEH_handler_aml_attributes_get (
  struct TEH_RequestContext *rc,
  const struct TALER_AmlOfficerPublicKeyP *officer_pub,
  const char *const args[])
{
  struct ResponseContext *rctx = rc->rh_ctx;
  int64_t limit = -20;
  uint64_t offset;
  struct TALER_NormalizedPaytoHashP h_payto;
  uint64_t file_number;

  if (NULL == rctx)
  {
    rctx = GNUNET_new (struct ResponseContext);
    rctx->rc = rc;
    rc->rh_ctx = rctx;
    rc->rh_cleaner = &free_rc;
  }
  else
  {
    if (NULL == rctx->response)
    {
      GNUNET_break (0);
      return MHD_NO;
    }
    return MHD_queue_response (rc->connection,
                               rctx->http_status,
                               rctx->response);
  }
  if ( (NULL == args[0]) ||
       (NULL != args[1]) )
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (
      rc->connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_GENERIC_ENDPOINT_UNKNOWN,
      rc->url);
  }
  if (GNUNET_OK !=
      GNUNET_STRINGS_string_to_data (args[0],
                                     strlen (args[0]),
                                     &h_payto,
                                     sizeof (h_payto)))
  {
    GNUNET_break_op (0);
    return TALER_MHD_reply_with_error (
      rc->connection,
      MHD_HTTP_BAD_REQUEST,
      TALER_EC_GENERIC_PATH_SEGMENT_MALFORMED,
      "h_payto");
  }

  TALER_MHD_parse_request_snumber (rc->connection,
                                   "limit",
                                   &limit);
  if (limit > 0)
    offset = 0;
  else
    offset = INT64_MAX;
  TALER_MHD_parse_request_number (rc->connection,
                                  "offset",
                                  &offset);

  {
    enum GNUNET_DB_QueryStatus qs;

    qs = TEH_plugin->lookup_aml_file_number (TEH_plugin->cls,
                                             &h_payto,
                                             &file_number,
                                             &rctx->is_wallet);
    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      return TALER_MHD_reply_with_error (
        rc->connection,
        MHD_HTTP_INTERNAL_SERVER_ERROR,
        TALER_EC_GENERIC_DB_FETCH_FAILED,
        "lookup_aml_file_number");
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      /* Account unknown, return 404 */
      return TALER_MHD_reply_with_error (
        rc->connection,
        MHD_HTTP_NOT_FOUND,
        TALER_EC_EXCHANGE_GENERIC_TARGET_ACCOUNT_UNKNOWN,
        args[0]);
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }
  }

  {
    const char *mime;

    mime = MHD_lookup_connection_value (rc->connection,
                                        MHD_HEADER_KIND,
                                        MHD_HTTP_HEADER_ACCEPT);
    if (NULL == mime)
      mime = "application/json";
    if (0 == strcmp (mime,
                     "application/json"))
    {
      rctx->format = RCF_JSON;
      rctx->details.json = json_array ();
      GNUNET_assert (NULL != rctx->details.json);
    }
    else if (0 == strcmp (mime,
                          "application/pdf"))
    {
      enum GNUNET_DB_QueryStatus qs;

      rctx->format = RCF_PDF;
      rctx->details.pdf.global_attrs = json_object ();
      GNUNET_assert (NULL != rctx->details.pdf.global_attrs);
      if (NULL != TEH_global_pdf_form_data)
        GNUNET_assert (0 ==
                       json_object_update (rctx->details.pdf.global_attrs,
                                           TEH_global_pdf_form_data));
      GNUNET_assert (0 ==
                     json_object_set_new (rctx->details.pdf.global_attrs,
                                          "FILE_NUMBER",
                                          json_integer (file_number)));
      /* Lookup latest AML decision & account rules & properties
         to build the cover page */
      qs = TEH_plugin->lookup_aml_history (TEH_plugin->cls,
                                           &h_payto,
                                           UINT64_MAX, /* offset */
                                           -1, /* latest decision only */
                                           &build_cover_page,
                                           rctx);
      switch (qs)
      {
      case GNUNET_DB_STATUS_HARD_ERROR:
      case GNUNET_DB_STATUS_SOFT_ERROR:
        GNUNET_break (0);
        return TALER_MHD_reply_with_error (
          rc->connection,
          MHD_HTTP_INTERNAL_SERVER_ERROR,
          TALER_EC_GENERIC_DB_FETCH_FAILED,
          "select_aml_attributes");
      case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
        /* no decision was ever taken, build empty cover page */
        GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                    "No decisions taken, creating empty cover page\n");
        build_cover_page (rctx,
                          0,
                          GNUNET_TIME_UNIT_ZERO_TS,
                          NULL,
                          NULL,
                          NULL,
                          NULL,
                          false,
                          false);
        break;
      case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
        break;
      }
    }
    else
    {
      GNUNET_break_op (0);
      return TALER_MHD_REPLY_JSON_PACK (
        rc->connection,
        MHD_HTTP_NOT_ACCEPTABLE,
        GNUNET_JSON_pack_string ("hint",
                                 mime));
    }
  }

  {
    enum GNUNET_DB_QueryStatus qs;

    if (limit > MAX_RECORDS)
      limit = MAX_RECORDS;
    if (limit < -MAX_RECORDS)
      limit = -MAX_RECORDS;
    qs = TEH_plugin->select_aml_attributes (
      TEH_plugin->cls,
      &h_payto,
      offset,
      limit,
      &detail_cb,
      rctx);
    GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                "Obtained %d/%d attributes\n",
                qs,
                (int) (limit < 0 ? -limit : limit));
    switch (qs)
    {
    case GNUNET_DB_STATUS_HARD_ERROR:
    case GNUNET_DB_STATUS_SOFT_ERROR:
      GNUNET_break (0);
      return TALER_MHD_reply_with_error (
        rc->connection,
        MHD_HTTP_INTERNAL_SERVER_ERROR,
        TALER_EC_GENERIC_DB_FETCH_FAILED,
        "select_aml_attributes");
    case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
      if (RCF_JSON == rctx->format)
        return TALER_MHD_reply_static (
          rc->connection,
          MHD_HTTP_NO_CONTENT,
          NULL,
          NULL,
          0);
      break;
    case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
      break;
    }
  }

  switch (rctx->format)
  {
  case RCF_JSON:
    return TALER_MHD_REPLY_JSON_PACK (
      rc->connection,
      MHD_HTTP_OK,
      GNUNET_JSON_pack_array_incref ("details",
                                     rctx->details.json));
  case RCF_PDF:
    /* Cover page: off of 0 is impossible */
    GNUNET_assert (0 != rctx->details.pdf.off);
    for (unsigned int i = 0; i<rctx->details.pdf.off; i++)
    {
      /* The jdata[i] could be a *string* in case of attachments,
         ignore those here! */
      if (NULL == rctx->details.pdf.docs[i].form_name)
      {
        GNUNET_assert (json_is_string (rctx->details.pdf.jdata[i]));
      }
      else
      {
        GNUNET_assert (json_is_object (rctx->details.pdf.jdata[i]));
        GNUNET_assert (0 ==
                       json_object_update_missing (
                         rctx->details.pdf.jdata[i],
                         rctx->details.pdf.global_attrs)
                       );
      }
    }
    rctx->tc = TALER_MHD_typst (TEH_cfg,
                                false,
                                "exchange",
                                rctx->details.pdf.off,
                                rctx->details.pdf.docs,
                                &pdf_cb,
                                rctx);
    if (NULL == rctx->tc)
    {
      GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                  "Client requested PDF, but Typst is unavailable\n");
      return TALER_MHD_reply_with_error (
        rc->connection,
        MHD_HTTP_NOT_IMPLEMENTED,
        TALER_EC_EXCHANGE_GENERIC_NO_TYPST_OR_PDFTK,
        NULL);
    }
    GNUNET_CONTAINER_DLL_insert (rctx_head,
                                 rctx_tail,
                                 rctx);
    MHD_suspend_connection (rc->connection);
    return MHD_YES;
  }
  GNUNET_assert (0);
  return MHD_NO;
}


/* end of taler-exchange-httpd_aml-attributes_get.c */
