use ui;
use layout;
use progvis:net;
use core:geometry;
use core:io;

/**
 * Dialog for uploading new problems.
 */
dialog UploadDlg {
	layout Grid {
		expandCol: 1;
		Label nameLbl("Name:", HAlign:right, VAlign:center) {}
		Edit name("Name your problem") { colspan: 2; }
		nextLine;
		Label implLbl("Implementation:", HAlign:right, VAlign:center) {}
		Edit implName("") {}
		Button implButton("Select...") {}
		nextLine;
		Label testLbl("Test program:", HAlign:right, VAlign:center) {}
		Edit testName("") {}
		Button testButton("Select...") {}
		nextLine;
		Label refLbl("Reference implementation:", HAlign:right, VAlign:center) {}
		Edit refName("") {}
		Button refButton("Select...") {}
		nextLine;
		Label("Additional errors to consider:", HAlign:right) {}
		CheckButton checkLeaks("Memory leaks") { colspan: 2; checked: true; }
		nextLine;
		CheckButton checkTermination("Livelocks") { col: 1; colspan: 2; checked: true; }
		nextLine;
		CheckButton checkBusyWait("Busy wait") { col: 1; colspan: 2; checked: true; }
		nextLine;
		CheckButton checkIgnoreKnownBusyWait("Ignore calls to \"intentional_busy_wait()\"") { col: 1; colspan: 2; }
		nextLine;
		Label("Acceptable errors for initial attempt:", HAlign:right) {}
		CheckButton acceptLeaks("Memory leaks") { colspan: 2; checked: true; }
		nextLine;
		CheckButton acceptTermination("Livelocks") { col: 1; colspan: 2; }
		nextLine;
		CheckButton acceptBusyWait("Busy wait") { col: 1; colspan: 2; checked: true; }
		nextLine;
		Progress progress { col: 0; colspan: 2; }
		Button uploadButton("Upload") { col: 2; }
	}

	Client client;

	// Upload a new problem.
	init(Client client) {
		init("Submit a problem", Size(500, 10)) {
			client = client;
		}

		progress.visible = false;
		defaultChoice(uploadButton);
		uploadButton.onClick = &this.upload();

		implButton.onClick = &this.pickImpl();
		testButton.onClick = &this.pickTest();
		refButton.onClick = &this.pickRef();
	}

	void pickImpl() {
		unless (f = pickFile())
			return;

		implName.text = f.format;

		Str title = f.title;
		Str ext = f.ext;

		Url test = f.parent / (title + "_test." + ext);
		Url ref = f.parent / (title + "_ref." + ext);
		if (test.exists())
			testName.text = test.format;
		if (ref.exists())
			refName.text = ref.format;
	}

	void pickTest() {
		unless (f = pickFile())
			return;

		testName.text = f.format;
	}

	void pickRef() {
		unless (f = pickFile())
			return;

		refName.text = f.format;
	}

	private Url? pickFile() {
		FileTypes ft("Source code");
		for (k in progvis:program:Program:supportedFiles)
			ft.add("${k}-source", [k]);

		var picker = FilePicker:open(ft);
		if (!picker.show(this))
			return null;

		picker.result;
	}

	void upload() {
		unless (name.text != "") {
			showMessage(this, "No name", "You need to specify a name for the problem.");
			return;
		}

		unless (impl = loadCode(implName.text)) {
			showMessage(this, "File not found", "The file for the implementation does not exist!");
			return;
		}

		unless (test = loadCode(testName.text)) {
			showMessage(this, "File not found", "The file for the test program does not exist!");
			return;
		}

		unless (ref = loadCode(refName.text)) {
			showMessage(this, "File not found", "The file for the reference implementation does not exist!");
			return;
		}

		progress.visible = true;
		progress.wait();
		for (x in children)
			x.enabled = false;

		ProblemChecks c(checkTermination.checked, checkLeaks.checked, checkBusyWait.checked, checkIgnoreKnownBusyWait.checked);
		AcceptableErrors a(acceptTermination.checked, acceptLeaks.checked, acceptBusyWait.checked);

		(spawn checkCode(name.text, test, impl, ref, c, a)).detach();
	}

	private progvis:net:Code? loadCode(Str path) {
		Url url = parsePath(path);
		if (!url.exists)
			return null;

		Str contents = url.readAllText;
		if (contents.empty)
			return null;

		return progvis:net:Code(contents, url.ext);
	}

	void checkCode(Str name, progvis:net:Code test, progvis:net:Code impl, progvis:net:Code ref,
				ProblemChecks checks, AcceptableErrors acceptable) {
		MemoryProtocol memory;
		Url m = test.put("test", memory);
		Url i = impl.put("impl", memory);
		Url r = ref.put("ref", memory);

		try {
			if (error = checks.check([m, r])) {
				error("The test program does not follow the specification from the reference solution.\nIt produced the following error: ${error.error.type}");
				return;
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the reference solution:\n" + e.message, e.pos);
			return;
		}

		Error currentError = Error:unknown();
		try {
			if (error = checks.check([m, i])) {
				currentError = Error:error(error.error.type);
			} else {
				currentError = Error:success();
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the implementation:\n" + e.message, e.pos);
			return;
		}

		try {
			client.query(NewProblemRequest(name, impl, test, ref, checks, acceptable, currentError));
		} catch (Exception e) {
			error(e.message);
			return;
		}

		done();
		close(1);
	}

	void error(Str message, core:lang:SrcPos pos) {
		done();
		progvis:CompileErrorDialog(message, pos).show(this);
	}

	void error(Str message) {
		done();
		showMessage(this, "Error", message);
	}

	void done() {
		progress.visible = false;
		for (x in children)
			x.enabled = true;
	}

}


/**
 * Dialog for uploading improved implementations or tests.
 */
dialog SolutionUploadDlg {
	layout Grid {
		expandCol: 1;
		Label status("Checking your solution...") { colspan: 3; }
		nextLine;
		Progress progress { col: 0; colspan: 2; }
		Button uploadButton("Submit") { col: 2; }
	}

	private Client client;
	private Problem original;
	private progvis:net:Code? newTest;
	private progvis:net:Code? newImpl;
	private Request? request;

	// Cancel auto-grading.
	private Bool cancel;

	init(Client client, Problem original, progvis:net:Code? newTest, progvis:net:Code? newImpl) {
		init("Submit your solution", Size(400, 10)) {
			client = client;
			original = original;
			newTest = newTest;
			newImpl = newImpl;
			cancel = false;
		}

		progress.wait();
		uploadButton.enabled = false;
		defaultChoice = uploadButton;
		uploadButton.onClick = &this.upload();

		(spawn checkCode()).detach();
	}

	void onDestroy(Int code) : override {
		cancel = true;
	}

	void upload() {
		unless (request)
			return;

		uploadButton.enabled = false;
		(spawn doUpload(request)).detach();
	}

	private void doUpload(Request request) {
		try {
			client.query(request);
			close(1);
		} catch (Exception e) {
			showMessage(this, "Error", e.message);
			uploadButton.enabled = true;
		}
	}

	void checkCode() {
		MemoryProtocol memory;
		Url m = (if (newTest) { newTest; } else { original.test; }).put("test", memory);
		Url i = (if (newImpl) { newImpl; } else { original.impl; }).put("impl", memory);
		Url r = original.ref.put("ref", memory);

		UiProgress progress(this);

		if (newTest.any) {
			try {
				if (error = original.checks.check([m, r], progress)) {
					error("The test program does not follow the specification of the problem.");
					return;
				}
			} catch (core:lang:CodeError e) {
				if (file = e.pos.file) {
					if (file == m) {
						error("Error while compiling your test program with the reference implementation:\n" + e.message, e.pos);
						return;
					}
				}
				error("Error while compiling your test program with the reference implementation:\n" + e.message);
			}
		}

		if (newImpl) {
			// Check if the new implementation is different from the original one.
			if (newImpl.signature == original.impl.signature) {
				error("You have not modified the implementation. Make some modifications and click \"Reload\" before submitting.");
				return;
			}
		}

		if (cancel)
			return;

		Error currentError = Error:unknown();
		try {
			if (error = original.checks.check([m, i])) {
				currentError = Error:error(error.error.type);
			} else {
				currentError = Error:success();
			}
		} catch (core:lang:CodeError e) {
			error("Error while compiling the implementation:\n" + e.message, e.pos);
			return;
		}

		if (newTest.any) {
			if (error = currentError.error) {
				done("You found an error of type ${error}, click \"Submit\" to proceed!");
			} else {
				error("Your test program did not find an error in the current implementation, try again!");
			}
			size = minSize();
		} else {
			if (error = currentError.error) {
				Bool refuseError = original.testId < 0;
				if (refuseError)
					refuseError = !original.acceptableErrors.check(error);

				if (refuseError) {
					// Require a solution initially.
					error("Your solution does not solve all errors with this test program.\nI found an error of type: ${error}\nYou need to solve this error for the initial problem. Try again!");
				} else {
					// We allow uploading the problems again, but with reduced reward. Thus, one student
					// may solve one issue, but another may solve another issue. We check so that the
					// code is not identical as the first part here.
					done("Your solution does not solve all errors with this test program.\nI found an error of type: ${error}\nYou may still upload your solution, but you will not receive full points for it.");
				}
			} else {
				done("You fixed the current issues, click \"Submit\" to proceed!");
			}
		}

		if (uploadButton.enabled) {
			if (newTest) {
				request = NewTest(original.id, newTest, original.implId, currentError);
			} else if (newImpl) {
				request = NewImplementation(original.id, newImpl, original.testId, currentError);
			}
		}
	}

	class UiProgress extends progvis:check:Progress {
		init(SolutionUploadDlg owner) {
			init { owner = owner; }
		}

		SolutionUploadDlg owner;

		void progress(Nat depth, Nat states, Nat edges, Nat unexplored) : override {
			if (owner.cancel)
				throw progvis:program:ExitError();
		}
	}

	private void error(Str message, core:lang:SrcPos pos) {
		error("Compilation error, try again.");
		progvis:CompileErrorDialog(message, pos).show(this);
	}

	void error(Str message) {
		status.text = message;
		progress.visible = false;
		size = minSize();
	}

	void done(Str message) {
		status.text = message;
		progress.visible = false;
		uploadButton.enabled = true;
		size = minSize();
	}

}
