#!/bin/sh
set -e
# Repack legacy uqm-russian (0.6) to UQM 0.8 format: base/ paths, comm renames, fonts, apply patches.
# Called from debian/rules.
# Usage: repack-content-0.8.0.sh [input.zip] [output_dir]
#  output_dir — where to put uqm-0.8.0-russian.uqm (default: debian/unpack).
# Patches and starcon-extra are read from the script directory (debian/).

DEBIAN_DIR="$(cd "$(dirname "$0")" && pwd)"
REPO_ROOT="$(cd "$DEBIAN_DIR/.." && pwd)"

INPUT_ZIP="${1:-}"
if [ -n "$INPUT_ZIP" ]; then
  [ ! -f "$INPUT_ZIP" ] && [ -f "$REPO_ROOT/$INPUT_ZIP" ] && INPUT_ZIP="$REPO_ROOT/$INPUT_ZIP"
else
  INPUT_ZIP="$(ls "$REPO_ROOT"/russian-*.zip 2>/dev/null | head -1)"
fi
if [ -z "$INPUT_ZIP" ] || [ ! -f "$INPUT_ZIP" ]; then
  echo "repack-content-0.8.0.sh: need input.zip (argument or russian-*.zip in root)." >&2
  exit 1
fi

OUTPUT_DIR="${2:+$REPO_ROOT/$2}"
OUTPUT_DIR="${OUTPUT_DIR:-$DEBIAN_DIR/unpack}"
OUTPUT_DIR="$(cd "$REPO_ROOT" && mkdir -p "$OUTPUT_DIR" && cd "$OUTPUT_DIR" && pwd)"
TMP_DIR="$OUTPUT_DIR/tmp-repack"
rm -rf "$TMP_DIR"
mkdir -p "$TMP_DIR"

# Unpack: outer zip may contain russian.zip with actual content
unzip -q "$INPUT_ZIP" -d "$TMP_DIR/outer"
# ChangeLog for dh_installchangelogs — in the old package it was at the outer zip root
[ -f "$TMP_DIR/outer/ChangeLog.txt" ] && cp "$TMP_DIR/outer/ChangeLog.txt" "$OUTPUT_DIR/ChangeLog.txt"
INNER_ZIP=""
for z in "$TMP_DIR/outer/russian.zip" "$TMP_DIR/outer"/*.zip; do
  [ -f "$z" ] && [ "$z" != "$INPUT_ZIP" ] && INNER_ZIP="$z" && break
done
if [ -n "$INNER_ZIP" ]; then
  unzip -q "$INNER_ZIP" -d "$TMP_DIR/legacy"
  rm -rf "$TMP_DIR/outer"
else
  mv "$TMP_DIR/outer" "$TMP_DIR/legacy"
fi
cd "$TMP_DIR/legacy"

# In 0.8 content lives under base/
mkdir -p base
for d in *; do
  [ "$d" = base ] && continue
  [ -d "$d" ] && mv "$d" base/
done

# UQM 0.7: comm dir and .txt renames
if [ -d base/comm ]; then
  cd base/comm
  rename_comm() {
    old="$1"; new="$2"
    if [ -d "$old" ]; then
      mv "$old" "$new"
      [ -f "$new/$old.txt" ] && mv "$new/$old.txt" "$new/$new.txt"
    fi
  }
  rename_comm thradd thraddash
  rename_comm slyland slylandro
  rename_comm comandr commander
  rename_comm starbas starbase
  rename_comm blackur kohrah
  rename_comm melnorm melnorme
  rename_comm rebel yehatrebels
  rename_comm talkpet talkingpet
  rename_comm shofixt shofixti
  rename_comm zoqfot zoqfotpik
  rename_comm spahome safeones
  cd "$TMP_DIR/legacy"
fi

# Fonts: from base/comm/* to base/fonts/, glyph names decimal→hex
if [ -d base/comm ]; then
  mkdir -p base/fonts
  for race_dir in base/comm/*/; do
    [ -d "$race_dir" ] || continue
    race="$(basename "$race_dir")"
    fon_src=""
    if [ -d "base/comm/$race/$race.fon" ]; then
      fon_src="base/comm/$race/$race.fon"
    elif [ "$race" = "commander" ] && [ -d "base/comm/commander/comandr.fon" ]; then
      fon_src="base/comm/commander/comandr.fon"
    fi
    if [ -n "$fon_src" ]; then
      case "$race" in
        commander) cp -a "$fon_src" "base/fonts/player.fon" ;;
        *)         cp -a "$fon_src" "base/fonts/$race.fon" ;;
      esac
    fi
  done
  [ -d base/lbm/micro.fon ] && cp -a base/lbm/micro.fon base/fonts/micro.fon
  [ -d base/lbm/starcon.fon ] && cp -a base/lbm/starcon.fon base/fonts/starcon.fon
  [ -d base/lbm/tiny.fon ] && cp -a base/lbm/tiny.fon base/fonts/tiny.fon
  for fon_dir in base/fonts/*.fon; do
    [ -d "$fon_dir" ] || continue
    cd "$fon_dir"
    for f in *.png; do
      [ -f "$f" ] || continue
      b="${f%.*}"
      case "$b" in
        [0-9]*) new=$(printf "%x.png" "$b"); [ "$f" != "$new" ] && mv "$f" "$new" ;;
      esac
    done
    for f in *.lbm; do
      [ -f "$f" ] || continue
      b="${f%.*}"
      case "$b" in
        [0-9]*) new=$(printf "%x.lbm" "$b"); [ "$f" != "$new" ] && mv "$f" "$new" ;;
      esac
    done
    cd "$TMP_DIR/legacy"
  done
fi

cd "$REPO_ROOT"

# gamestrings: starcon.txt + starcon-extra → base/gamestrings.txt, then apply patch
if [ -f "$TMP_DIR/legacy/base/lbm/starcon.txt" ] && [ -f "$DEBIAN_DIR/starcon-extra.txt" ]; then
  cat "$TMP_DIR/legacy/base/lbm/starcon.txt" "$DEBIAN_DIR/starcon-extra.txt" > "$TMP_DIR/legacy/base/gamestrings.txt"
  if [ -f "$DEBIAN_DIR/patches/gamestrings.patch" ]; then
    (cd "$TMP_DIR/legacy" && patch -p1 < "$DEBIAN_DIR/patches/gamestrings.patch") || true
  fi
fi

# setupmenu: replace with our setupmenu-ru.txt (0.8 loads from base/ui/setupmenu.txt)
if [ -f "$DEBIAN_DIR/setupmenu-ru.txt" ]; then
  mkdir -p "$TMP_DIR/legacy/base/ui"
  cp "$DEBIAN_DIR/setupmenu-ru.txt" "$TMP_DIR/legacy/base/ui/setupmenu.txt"
fi

# main menu (New Game / Load Game / …): newgame-000..005.png + newgame.ani → base/ui/
if [ -f "$DEBIAN_DIR/startmenu-ru/newgame-000.png" ]; then
  mkdir -p "$TMP_DIR/legacy/base/ui"
  cp "$DEBIAN_DIR/startmenu-ru/newgame-000.png" "$DEBIAN_DIR/startmenu-ru/newgame-001.png" \
     "$DEBIAN_DIR/startmenu-ru/newgame-002.png" "$DEBIAN_DIR/startmenu-ru/newgame-003.png" \
     "$DEBIAN_DIR/startmenu-ru/newgame-004.png" "$DEBIAN_DIR/startmenu-ru/newgame-005.png" \
     "$TMP_DIR/legacy/base/ui/"
  [ -f "$DEBIAN_DIR/startmenu-ru/newgame.ani" ] && cp "$DEBIAN_DIR/startmenu-ru/newgame.ani" "$TMP_DIR/legacy/base/ui/"
fi

# Build addon: shadow-content zip + .rmp
mkdir -p "$TMP_DIR/russian/shadow-content"
(cd "$TMP_DIR/legacy" && zip -r -q -0 "$TMP_DIR/russian/shadow-content/russian-content.zip" base/)
printf '# Russian translation addon - overrides base via shadow-content\n' > "$TMP_DIR/russian/russian.rmp"
(cd "$TMP_DIR" && zip -r -q -0 "$OUTPUT_DIR/uqm-0.8.0-russian.uqm" russian/)
# ChangeLog for dh_installchangelogs, if present in sources
for cl in "$TMP_DIR/legacy/ChangeLog.txt" "$TMP_DIR/legacy/base/ChangeLog.txt" "$TMP_DIR/outer"/*/ChangeLog.txt; do
  [ -f "$cl" ] && cp "$cl" "$OUTPUT_DIR/ChangeLog.txt" && break
done
rm -rf "$TMP_DIR"

echo "repack-content-0.8.0.sh: $OUTPUT_DIR/uqm-0.8.0-russian.uqm created."
